// Copyright (C) 2015-2017 Bayerische Motoren Werke Aktiengesellschaft (BMW AG)
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

///* steps to get this server working on the target
///
/// 1. export VSOMEIP_CONFIGURATION=[path]/vsomeipUdpServer.json  -> this is telling where the configuration file is located
/// 2. export VSOMEIP_APPLICATION_NAME = hello_world_client  -> this is telling what the server application is. The name is refered in the jone file
/// 3. switch off the firewall
/// 4. sudo route add -nv 224.224.224.245 dev eth1   -> this is required for service discovery
/// 5. VLAN setup (configuration files are available in config folder
/// 6. run someip_gateway_service_out.out
//#include <unistd.h>
#include <iostream>
#include <time.h>
#include "someip_Gateway.h"
#include "SomeIPMessageObserverItf.h"
#include "someipDefines.h"
#include <thread>
#include <cstring>
#include <cstdint>
#include "../app/Macro.h"

#include "../app/SleepTimer.h"
#include "dlt/dlt.h"
DLT_IMPORT_CONTEXT (SOMEIP);

//____________________________________________________________________________________
// map
static vsomeip::service_t map_cluster_service_id = CLUSTER_VIDEO_SERVICE_ID;
static vsomeip::instance_t map_cluster_instance_id = CLUSTER_VIDEO_INSTANCE_ID;
static vsomeip::method_t service_method_id = CLUSTER_SERVICE_METHOD_ID;
static vsomeip::method_t Event_VideoStreamStatusId = CLUSTER_VIDEO_STREAM_STATUS_EVENT_ID; // tell client the status of video stream
static vsomeip::method_t ServiceMethod_Request_LayoutId = CLUSTER_VIDEO_LAYOUT_REQUEST_METHOD_ID;   // client requests a new layout
#define VideoStreamStatusEventID Event_VideoStreamStatusId

//____________________________________________________________________________________
// swipe
static vsomeip::service_t CombinationServiceID = CLUSTER_ANIMATED_SWIPE_SERVICE_ID; // AIVI combination (swipe)
static vsomeip::instance_t ClusterCombinationInstanceID = CLUSTER_ANIMATED_SWIPE_INSTANCE_ID;
static vsomeip::method_t ClusterCombination_ResponseMessageID = CLUSTER_ANIMATED_SWIPE_METHOD_ID;

#define DLTCOMMANDID 0x9090
#define DLT_COMMAND_STOPOFFER 0x0001
#define DLT_COMMAND_STARTOFFER 0x0002
#define DLT_COMMAND_PROCESSIMAGE 0x0003
#define DLT_COMMAND_STARSTREAM 0x0004
#define DLT_COMMAND_ANIMATED_SWIPE 0x0005
#define DLT_COMMAND_SWIPE_RESPONSE 0x0006
#define DLT_COMMAND_SEND_LAYOUT_STATUS 0x0007
#define DLT_COMMAND_CLUSTER_COMBINATION_AVAILABILITY 0x0008


SomeIPGateway* SomeIPGateway::pSomeIPGateway=nullptr;
// Map in CLuster Service
static vsomeip::eventgroup_t MapInClusterEventGroup = CLUSTER_VIDEO_STREAM_STATUS_EVENT_GROUP_ID;
// Media Image Service
#define Media_image_Groupe_id 1
static vsomeip::service_t picture_cluster_service_id = CLUSTER_PICTURE_SERVICE_ID;
static vsomeip::eventgroup_t picture_cluster_event_group = CLUSTER_PICTURE_EVENT_GROUP_ID;
static vsomeip::instance_t picture_cluster_instance_id = CLUSTER_PICTURE_INSTANCE_ID;
static vsomeip::event_t AudioPictureUpdatesId = CLUSTER_PICTURE_AUDIO_PICTURE_UPDATE_EVENT_ID;
static vsomeip::event_t PhonePictureUpdatesId = CLUSTER_PICTURE_PHONE_PICTURE_UPDATE_EVENT_ID;
ThreadAdministration SomeIPGateway::oObserverThread; //this thread is waiting for "send event to cluster".
someIP_MessageSender SomeIPGateway::oSomeIPMessageSender;

// ToDO: this function should be implemented in SomeIpMessageSender, but this is currently not working.
bool SomeIPGateway::sendOldestPendingEventToCluster() {
    std::cout << "SomeIPGateway::sendOldestPendingEventToCluster(): enter: amount of pending event waiting to be sent: "<<  oSomeIPMessageSender.oEventQueue.size()<< std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendOldestPendingEventToCluster(): enter: amount of pending event waiting to be sent:"), DLT_INT(oSomeIPMessageSender.oEventQueue.size()));
    bool AllMessagesSent = false; // at the moment we assume, that data are not going to be written and read at the same time.
    short StreamStatus = 0; // hard coded value is fine here

    // ToDo: this if case should be replace by a function e.g. "getEventAndSentIt", but not done since sendOldestPendingEventToCluster is not implemented in SomeIPMessageSender

    if (false == oSomeIPMessageSender.oEventQueue.isTableEmpty()) {
        std::cout << "SomeIPGateway::sendOldestPendingEventToCluster(): status: " << StreamStatus << " is pending, try sending it" << std::endl;
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendOldestPendingEventToCluster(): status:"), DLT_INT(StreamStatus), DLT_STRING(" is pending, try sending it"));
        StreamStatus = oSomeIPMessageSender.getOldestElementAndRemoveItFromQueue();

        sendEvent(CLUSTER_VIDEO_SERVICE_ID, CLUSTER_VIDEO_INSTANCE_ID, CLUSTER_VIDEO_STREAM_STATUS_EVENT_ID, StreamStatus); // ToDo: oEventQueue must store the complete data set. It stores the status only at the moment, thus all other are hard coded.
        std::cout << "SomeIPGateway::sendOldestPendingEventToCluster(): event: amount of pending event waiting to be sent: "<<  oSomeIPMessageSender.oEventQueue.size() << std::endl;
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendOldestPendingEventToCluster(): event: amount of pending event waiting to be sent:"), DLT_INT(oSomeIPMessageSender.oEventQueue.size()));
    }
    else
    {
        std::cout << "SomeIPGateway::sendOldestPendingEventToCluster(): nothing is pending"  << std::endl;
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendOldestPendingEventToCluster(): nothing is pending"));
    }

    if (true == oSomeIPMessageSender.oEventQueue.isTableEmpty())
    {
        AllMessagesSent = true;
    }
    std::cout << "SomeIPGateway::sendOldestPendingEventToCluster(): exit: all sent:" << AllMessagesSent << " messages in queue: " << oSomeIPMessageSender.oEventQueue.size() << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendOldestPendingEventToCluster(): exit: all sent:"), DLT_INT(AllMessagesSent), DLT_STRING(" messages in queue: "), DLT_INT(oSomeIPMessageSender.oEventQueue.size() ));
    return (AllMessagesSent);
}

// once the observer is receiving a message, it'll call the consumer thread. This function is embedded in the consumer thread.
// even though return value is not given, we need to define the function exactly this way, because it's been used as function pointer!
void* SomeIPGateway::MessageConsumer(void* Data) {
    std::cout << "SomeIPGateway::MessageConsumer: enter" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::MessageConsumer: enter"));
    bool AllMessageReadOfQueue = false;
    std::cout << "SomeIPGateway::MessageConsumer: enter" << oObserverThread.terminateObserverThread << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::MessageConsumer: enter"));
    while (false == oObserverThread.terminateObserverThread) {
        if (true == oObserverThread.waitForNewMessagesIncoming()) {
            std::cout << "SomeIPGateway::MessageConsumer: check queue for pending event and send the same" << std::endl;
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::MessageConsumer: check queue for pending event and send the same"));
            AllMessageReadOfQueue = false;
            while (false == AllMessageReadOfQueue) {
                AllMessageReadOfQueue = sendOldestPendingEventToCluster();
                std::cout << "SomeIPGateway::MessageConsumer: all read?"<< AllMessageReadOfQueue << std::endl;
                DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::MessageConsumer: all read?"), DLT_INT(AllMessageReadOfQueue));
                sleep(1); //ToDo: Avoid endless loop reset. This is temporary for safety at the moment. Further on, timeout time influence also the events sending. Without sleep not all events are sent to Cluster. I think it's because a pending event is going to be overwritten by a new one.
            }
        }
        std::cout << "SomeIPGateway::MessageConsumer: wait for new events" << std::endl;
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::MessageConsumer: wait for new events"));
    }
    std::cout << "SomeIPGateway::MessageConsumer: leaving thread now: don't wait for events any more" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::MessageConsumer: leaving thread now: don't wait for events any more"));
    void* RetDummy = nullptr;
    return (RetDummy); // this is required for new compiler switches. The function definition can't be changed, becaue this is a callback function.
}

/**
 * create the message consumer thread. This is waiting for signals and will send something to cluster if right signal is been received.
 * @return: true: thread is been created
 */
bool SomeIPGateway::createMessageConsumer() {
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::createMessageConsumer(): entered"));
    std::cout << "SomeIPGateway::createMessageConsumer: enter" << std::endl;
    bool threadCreatePassed = true;

    if (oObserverThread.createConsumerThread(&MessageConsumer) != 0) {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("Error: SomeIPGateway Consumer Thread create failed!"));
        std::cout << "Error: SomeIPGateway Consumer Thread create failed!" << std::endl;

        exit (EXIT_FAILURE);
        threadCreatePassed = FALSE;
    } else {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway Consumer Thread create passed!"));
        std::cout << "SomeIPGateway Consumer Thread create passed!" << std::endl;
    }

    return (threadCreatePassed);

}

/**
 * configure the object which will send messages to Cluster.
 */
void SomeIPGateway::configureMessageSender()
{
    // ToDo: null point checks missing
    if ((nullptr != rtm_) && (nullptr != app_)) {
        std::cout << "SomeIPGateway::configureMessageSender: set Message sender: rtm address:" << rtm_ << " app address: " << app_ << std::endl;
//        oSomeIPMessageSender._SomeIPRuntime = rtm_;  // not used at the moment. Event are sent from here and not from SOmeIPMessageSender at the moment.
//        oSomeIPMessageSender._SomeIPApp = app_;// not used at the moment. Event are sent from here and not from SOmeIPMessageSender at the moment.
        oSomeIPMessageSender._ServiceID = map_cluster_service_id;
        oSomeIPMessageSender._ServiceInstanceID = map_cluster_instance_id;
        oSomeIPMessageSender._EventID = VideoStreamStatusEventID;
//        oSomeIPMessageSender.pSomeIPGateway = pSomeIPGateway;// not used at the moment. Event are sent from here and not from SOmeIPMessageSender at the moment.
    }
    else
    {
        std::cout << "SomeIPGateway::configureMessageSender: NULL Pointer exception !!!" << std::endl;
    }
}
// Get the vSomeIP runtime and
// create a application via the runtime, we could pass the application name
// here otherwise the name supplied via the VSOMEIP_APPLICATION_NAME
// environment variable is used
SomeIPGateway::SomeIPGateway() :
        _pLayoutRequestHandlerReference(nullptr),
        _pCombinationResponseHandlerReference(nullptr),
        _pCombinationServiceAvailabilityHandlerReference(nullptr),
        rtm_(vsomeip::runtime::get()), app_(rtm_->create_application()), stop_(false), stop_thread_(
                std::bind(&SomeIPGateway::stop, this)) {
    DLT_REGISTER_APP("SIAG", "SomeIP DLT Identifier for Logging");
    std::cout << "SomeIPGateway::constructor: enter" << std::endl;

#if defined (CLUSTER_EVENT_HANDLING_USE_THREAD_FOR_SENDING_EVENTS)
    SomeIPGateway::createMessageConsumer();
#endif
    configureMessageSender();
    std::cout << "SomeIPGateway::constructor: exit" << std::endl;
    SomeIPApplicationIsRegistered = false;
    ServiceOffer_MapInClusterIsOffered = false;
    EventSubscription_MapInCLusterClientIsRegisteredForEvent = false;
    EventSubscription_ImageInCLusterClientIsRegisteredForEvent = false;
    ServiceAvailable_ClusterCombinationAvailable = false;
    oLayoutRequestObserver = NULL;
    m_MediaImageExecutorReference = NULL;
    bEventNotSent = FALSE;
    
}

SomeIPGateway::~SomeIPGateway() {
    std::cout << "SomeIPGateway::destructor: enter" << std::endl;
    stop_thread_.join();
    oObserverThread.waitConsumerThreadEnd();
    oObserverThread.exitMainThread();
}

// _____________________________________________________________________
// connect the handler
// the handler will deal with the messages received from Cluster
void SomeIPGateway::setLayoutRequestHandler(SomeIPMessageObserverItf* HandlerReference) {
    std::cout << "SomeIPGateway::setLayoutRequestHandler: enter" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::setLayoutRequestHandler: entered"));
    if (nullptr != HandlerReference) {
        _pLayoutRequestHandlerReference = HandlerReference;
        oLayoutRequestMessageObserver.setLayoutRequestHandler(HandlerReference);
    }
    else
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::setLayoutRequestHandler: handler not defined"));
    }
}

void SomeIPGateway::setClusterFadeInStatusHandler(SomeIPMessageObserverItf* HandlerReference) {
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::setClusterFadeInStatusHandler: entered"));
    if (nullptr != HandlerReference) {
        _pCombinationResponseHandlerReference = HandlerReference;
    }
    else
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::setClusterFadeInStatusHandler: handler not defined"));
    }
}

void SomeIPGateway::setClusterCombinationServiceAvailabilityHandler(SomeIPMessageObserverItf* HandlerReference)
{
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::setClusterCombinationServiceAvailabilityHandler: entered"));
    if (nullptr != HandlerReference) {
        _pCombinationServiceAvailabilityHandlerReference = HandlerReference;
    }
    else
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::setClusterCombinationServiceAvailabilityHandler: handler not defined"));
    }
}

// _____________________________________________________________________
// dlt test commands
int SomeIPGateway::uIntDLTTraceInjectionCallback(unsigned int service_id, void *data, unsigned int length)
{
    std::cout << "SomeIPGateway::uIntDLTTraceInjectionCallback: enter" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("Entered inside the uIntDLTTraceInjectionCallback"));
    const unsigned char  *Rxdata;
    Rxdata = static_cast<const unsigned char*>(data);
    tU32 u32MsgCode = (Rxdata[0]);
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("Injected command with code: "), DLT_INT((int)u32MsgCode));
    switch(u32MsgCode)
    {
    case DLT_COMMAND_STOPOFFER:
        pSomeIPGateway->stopOfferService();
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: Called stopOfferService"));
        break;
    case DLT_COMMAND_STARTOFFER:
        pSomeIPGateway->offerService();
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: Called startOfferService"));
        break;
    case DLT_COMMAND_PROCESSIMAGE:
        if (nullptr != pSomeIPGateway->m_MediaImageExecutorReference)
        {
            pSomeIPGateway->m_MediaImageExecutorReference->processTestMediaImage();
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: Called processTestMediaImage"));
        } else {
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: MediaImageExecutorReference is null"));
        }
        break;
    case DLT_COMMAND_STARSTREAM:
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: Called startStream"));
        if (length >= 2) {
            tU32 u32MsgParam1 = (Rxdata[1]);
            if (nullptr != pSomeIPGateway->_pLayoutRequestHandlerReference) {
                pSomeIPGateway->_pLayoutRequestHandlerReference->NewDataReceivedFromClient((short) u32MsgParam1);
            } else {
                DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("NULL pointer exception: Layout Request Handler is not defined !!!"));
            }

        } else {
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: not enough parameter!"));
        }
        break;
    case DLT_COMMAND_ANIMATED_SWIPE:
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: DLT_COMMAND_ANIMATED_SWIPE"));
        SOMEIP_NULL_POINTER_CHECK(pSomeIPGateway);
        pSomeIPGateway->sendIVIActionCombination(Rxdata[1]);
        break;
    }
    case DLT_COMMAND_SWIPE_RESPONSE:
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: DLT_COMMAND_SWIPE_RESPONSE"));
        SOMEIP_NULL_POINTER_CHECK(pSomeIPGateway);
        SOMEIP_NULL_POINTER_CHECK(pSomeIPGateway->_pCombinationResponseHandlerReference);
        pSomeIPGateway->_pCombinationResponseHandlerReference->NewDataReceivedFromClient(Rxdata[1]);
        break;
    }
    case DLT_COMMAND_SEND_LAYOUT_STATUS:
    {
        sendEvent(CLUSTER_VIDEO_SERVICE_ID, CLUSTER_VIDEO_INSTANCE_ID, CLUSTER_VIDEO_STREAM_STATUS_EVENT_ID, (short) Rxdata[1]);
        break;
    }
    case DLT_COMMAND_CLUSTER_COMBINATION_AVAILABILITY:
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::uIntDLTTraceInjectionCallback: DLT_COMMAND_CLUSTER_COMBINATION_AVAILABILITY"));
        SOMEIP_NULL_POINTER_CHECK(pSomeIPGateway);
        SOMEIP_NULL_POINTER_CHECK(pSomeIPGateway->_pCombinationServiceAvailabilityHandlerReference);
        pSomeIPGateway->_pCombinationServiceAvailabilityHandlerReference->NewDataReceivedFromClient(Rxdata[1]);
        break;
    }
    default:
        break;
    }
    return 0;
}

// _____________________________________________________________________
// the vsomeip stack related functions
bool SomeIPGateway::init() {
    // init the application
    std::cout << "SomeIPGateway::init: enter" << std::endl;
    DLT_REGISTER_INJECTION_CALLBACK(SOMEIP, DLTCOMMANDID, uIntDLTTraceInjectionCallback);
    if (!app_->init()) {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::init(): Couldn't initialize application"));
        return false;
    }
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::init(): Registering message handler"));


    EventSubscription_MapInCLusterClientIsRegisteredForEvent = false;
    EventSubscription_ImageInCLusterClientIsRegisteredForEvent = false;
    // ______________________________________________________________________
    // map
    // register a message handler callback for messages sent to our service
    app_->register_message_handler(map_cluster_service_id, map_cluster_instance_id, ServiceMethod_Request_LayoutId,
            std::bind(&SomeIPGateway::onLayoutRequestNoReturnMessage, this, std::placeholders::_1));

    // ______________________________________________________________________
    // swipe
    // register a callback which is called as soon as the service is available
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::init(): register availability handler for swipe: Service ID"),
            DLT_INT(CLUSTER_ANIMATED_SWIPE_SERVICE_ID),
            DLT_STRING(" InstanceID"), DLT_INT(CLUSTER_ANIMATED_SWIPE_INSTANCE_ID),
            DLT_STRING(" Major Version"), DLT_INT(MAJOR_VERSION),
            DLT_STRING(" Minor Version"), DLT_INT(MINOR_VERSION)
            );
    app_->register_availability_handler(CombinationServiceID, ClusterCombinationInstanceID,
            std::bind(&SomeIPGateway::on_availability_cbk_forSwipe, this,
                    std::placeholders::_1, std::placeholders::_2,
                    std::placeholders::_3), MAJOR_VERSION, MINOR_VERSION);

    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::init(): Registering message handler for swipe response from cluster: Service ID"),
            DLT_INT(CombinationServiceID),
            DLT_STRING(" InstanceID"), DLT_INT(ClusterCombinationInstanceID),
            DLT_STRING(" Response Message ID"), DLT_INT(ClusterCombination_ResponseMessageID)
            );

    app_->register_message_handler(CombinationServiceID, ClusterCombinationInstanceID, ClusterCombination_ResponseMessageID,
                std::bind(&SomeIPGateway::onClusterCombinationResponseMessage, this, std::placeholders::_1));

    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::init(): Registering state handler"));

    // _______________________________________________________________________
    // someip framework
    // register a state handler to get called back after registration at the
    // runtime was successful
    app_->register_state_handler(std::bind(&SomeIPGateway::on_state_cbk, this, std::placeholders::_1));

    // _______________________________________________________________________
    // events for map in cluster
    std::set < vsomeip::eventgroup_t > EventGroups;
    EventGroups.insert(MapInClusterEventGroup);
    offerEvent(map_cluster_service_id, map_cluster_instance_id, VideoStreamStatusEventID, EventGroups);

    // _______________________________________________________________________
    // events for image in cluster
    std::set < vsomeip::eventgroup_t > PictureClusterEventGroups;
    PictureClusterEventGroups.insert(picture_cluster_event_group);
    offerEvent(picture_cluster_service_id, picture_cluster_instance_id, AudioPictureUpdatesId, PictureClusterEventGroups);
    offerEvent(picture_cluster_service_id, picture_cluster_instance_id, PhonePictureUpdatesId, PictureClusterEventGroups);

    /**
     * This call back will be called any time a client registered or unregister for event group
     */
    vsomeip::subscription_handler_t EventSubscriptionHandlerMap([&](vsomeip::client_t client_t, bool reg) {
        std::cout << "SomeIPGateway::EventSubscriptionHandlerMap: enter: subscription for client: "<< client_t << " is client registered:" << reg << std::endl;
        DLT_LOG(SOMEIP, DLT_LOG_INFO,
                DLT_CSTRING("SomeIPGateway::EventSubscriptionHandlerMap: enter: subscription for client: "),
                DLT_INT(client_t),
                DLT_STRING("is client registered:"),
                DLT_INT(reg)
                );
        if(reg)
            {
            EventSubscription_MapInCLusterClientIsRegisteredForEvent = true;
            }
        else
            {
            EventSubscription_MapInCLusterClientIsRegisteredForEvent = false;
            }

        return true;
    });

    vsomeip::subscription_handler_t EventSubscriptionHandlerImage([&](vsomeip::client_t client_t, bool reg) {
        std::cout << "SomeIPGateway::EventSubscriptionHandlerImage: enter: subscription for client: "<< client_t << " is client registered:" << reg << std::endl;
        DLT_LOG(SOMEIP,
                DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::EventSubscriptionHandlerImage: enter: subscription for client: "),
                DLT_INT(client_t),
                DLT_STRING("is client registered:"),
                DLT_INT(reg)
                );
        if(reg) {
            EventSubscription_ImageInCLusterClientIsRegisteredForEvent = true;
            // m_MediaImageExecutorReference->processTestMediaImage();
        }
        else
            {
            EventSubscription_ImageInCLusterClientIsRegisteredForEvent = false;
            }
        return true;
    });

    vsomeip::subscription_handler_t EventSubscriptionHandlerPicture([&](vsomeip::client_t client_t, bool reg) {
        std::cout << "SomeIPGateway::EventSubscriptionHandler: enter: subscription for client: "<< client_t << std::endl;
        if(reg) {
            std::cout << "SomeIPGateway::EventSubscriptionHandler: enter: subscription for client registered" << std::endl;
            // m_MediaImageExecutorReference->processTestMediaImage();
        }
        return true;
    });

    app_->register_subscription_handler(map_cluster_service_id, map_cluster_instance_id, MapInClusterEventGroup, EventSubscriptionHandlerMap);

    app_->register_subscription_handler(picture_cluster_service_id, picture_cluster_instance_id, picture_cluster_event_group, EventSubscriptionHandlerImage);
    requestService();

    condition_.notify_one();

    SomeIPApplicationIsRegistered = false;
    ServiceOffer_MapInClusterIsOffered = false;
    ServiceAvailable_ClusterCombinationAvailable = false;
    return true;
}

void SomeIPGateway::start() {
    // start the application and wait for the on_event callback to be called
    // this method only returns when app_->stop() is called
    std::cout << "SomeIPGateway::start: enter" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::start(): entered"));
    app_->start();
}

void SomeIPGateway::stopOfferService() {
    // Stop offering the service
    app_->stop_offer_service(map_cluster_service_id, map_cluster_instance_id, MAJOR_VERSION, MINOR_VERSION);
    ServiceOffer_MapInClusterIsOffered=false;
    app_->stop_offer_service(picture_cluster_service_id, picture_cluster_instance_id, MAJOR_VERSION, MINOR_VERSION);
}

void SomeIPGateway::stop() {
    std::cout << "SomeIPGateway::stop: enter" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO,
            DLT_CSTRING("SomeIPGateway::stop(): entered"));
    std::unique_lock < std::mutex > its_lock(mutex_);
    while (!stop_) {
        condition_.wait(its_lock);
    }
    DLT_LOG(SOMEIP, DLT_LOG_INFO,
            DLT_CSTRING("SomeIPGateway::stop(): active waiting ended!"));
    condition_.notify_one();
    notify_condition_.notify_one();
    stopOfferService();
//    std::this_thread::sleep_for(std::chrono::seconds(5));
    // unregister the state handler
    app_->unregister_state_handler();
    // unregister the subscription handler
    app_->unregister_subscription_handler(map_cluster_service_id, map_cluster_instance_id, MapInClusterEventGroup);
    app_->unregister_subscription_handler(picture_cluster_service_id, picture_cluster_instance_id, picture_cluster_event_group);
    // unregister the message handler
    app_->unregister_message_handler(map_cluster_service_id, map_cluster_instance_id, service_method_id);
    app_->unregister_message_handler(CombinationServiceID, ClusterCombinationInstanceID, ClusterCombination_ResponseMessageID);
    // shutdown the application
    app_->stop();
    app_->clear_all_handler();
}

void SomeIPGateway::terminate() {
    std::cout << "SomeIPGateway::terminate: enter" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::terminate(): entered"));
    std::lock_guard < std::mutex > its_lock(mutex_);
    stop_ = true;
    condition_.notify_one();
    notify_condition_.notify_one();
    stopOfferService();
    app_->stop();
    app_->clear_all_handler();
}

void SomeIPGateway::offerService()
{
    // we are registered at the runtime and can offer our service
    //_____________________________________________________________________________________________________________
    // map
    app_->offer_service(map_cluster_service_id, map_cluster_instance_id, MAJOR_VERSION, MINOR_VERSION);
    ServiceOffer_MapInClusterIsOffered = true;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::offerService(): Server: registered and offer service ServiceID: "), DLT_INT(map_cluster_service_id),
            DLT_CSTRING(" ServiceInstance: "), DLT_INT(map_cluster_instance_id));
    std::cout << "Server: registered and offer service ServiceID: " << std::hex << map_cluster_service_id << " ServiceInstance: "
            << map_cluster_instance_id << std::endl;

    //_____________________________________________________________________________________________________________
    // picture
    app_->offer_service(picture_cluster_service_id, picture_cluster_instance_id, MAJOR_VERSION, MINOR_VERSION);
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::offerService(): Server: registered and offer service ServiceID: "), DLT_INT(picture_cluster_service_id),
            DLT_CSTRING(" ServiceInstance: "), DLT_INT(picture_cluster_instance_id));
    std::cout << "Server: registered and offer service ServiceID: " << std::hex << picture_cluster_service_id << " ServiceInstance: "
            << picture_cluster_instance_id << std::endl;

   if(bEventNotSent)
   {
      DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::offerService(): Server: registered and offer service ServiceID: "), DLT_INT(_ServiceID),
            DLT_CSTRING(" ServiceInstance: "), DLT_INT(_ServiceInstanceID));
      std::cout << "Server: registered and offer service ServiceID: " << std::hex << _ServiceID << " ServiceInstance: "  << _ServiceInstanceID << std::endl;
      if(_ServiceID == picture_cluster_service_id)
      {
        sendEvent(_ServiceID, _ServiceInstanceID, _EventID, _oData);
      }
   }

}

void SomeIPGateway::requestService()
{
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::requestService(): enter: Service ID"),
            DLT_INT(CombinationServiceID),
            DLT_STRING(" InstanceID"), DLT_INT(ClusterCombinationInstanceID),
            DLT_STRING(" Major Version"), DLT_INT(MAJOR_VERSION),
            DLT_STRING(" Minor Version"), DLT_INT(MINOR_VERSION)
            );
    app_->request_service(CombinationServiceID, ClusterCombinationInstanceID, MAJOR_VERSION, MINOR_VERSION);
}

void SomeIPGateway::offerEvents()
{
    // events for map in cluster
    std::set < vsomeip::eventgroup_t > MapEventGroups;
    MapEventGroups.insert(MapInClusterEventGroup);
    offerEvent(map_cluster_service_id, map_cluster_instance_id, VideoStreamStatusEventID, MapEventGroups);

    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::offerEvents(): Server: registered and offer event ServiceID: "), DLT_INT(map_cluster_service_id),
            DLT_CSTRING(" ServiceInstance: "), DLT_INT(map_cluster_instance_id),
            DLT_CSTRING(" Event ID: "), DLT_INT(VideoStreamStatusEventID)
//            DLT_CSTRING(" Event Groups: "), DLT_INT(MapEventGroups)
            );
    std::cout << "Server: registered and offer events ServiceID: " << std::hex << map_cluster_service_id << " ServiceInstance: "
            << map_cluster_instance_id << " Event ID:" << VideoStreamStatusEventID /*<< " Event Groups: " << MapEventGroups */<< std::endl;

    // events for image in cluster
    std::set < vsomeip::eventgroup_t > PictureClusterEventGroups;
    PictureClusterEventGroups.insert(picture_cluster_event_group);
    offerEvent(picture_cluster_service_id, picture_cluster_instance_id, AudioPictureUpdatesId, PictureClusterEventGroups);

    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::offerEvents(): Server: registered and offer event ServiceID: "), DLT_INT(picture_cluster_service_id),
            DLT_CSTRING(" ServiceInstance: "), DLT_INT(picture_cluster_instance_id),
            DLT_CSTRING(" Event ID: "), DLT_INT(AudioPictureUpdatesId)
//            DLT_CSTRING(" Event Groups: "), DLT_INT(PictureClusterEventGroups)
            );
    std::cout << "Server: registered and offer events ServiceID: " << std::hex << picture_cluster_service_id << " ServiceInstance: "
            << picture_cluster_instance_id << " Event ID:" << AudioPictureUpdatesId /*<< " Event Groups:" << PictureClusterEventGroups */<< std::endl;


    offerEvent(picture_cluster_service_id, picture_cluster_instance_id, PhonePictureUpdatesId, PictureClusterEventGroups);

    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::offerEvents(): Server: registered and offer event ServiceID: "), DLT_INT(picture_cluster_service_id),
            DLT_CSTRING(" ServiceInstance: "), DLT_INT(picture_cluster_instance_id),
            DLT_CSTRING(" Event ID: "), DLT_INT(PhonePictureUpdatesId)
//            DLT_CSTRING(" Event Groups: "), DLT_INT(PictureClusterEventGroups)
            );
    std::cout << "Server: registered and offer events ServiceID: " << std::hex << picture_cluster_service_id << " ServiceInstance: "
            << picture_cluster_instance_id << " Event ID:" << PhonePictureUpdatesId /*<< " Event Groups:" << PictureClusterEventGroups */<< std::endl;

}

void SomeIPGateway::on_availability_cbk_forSwipe(vsomeip::service_t _service, vsomeip::instance_t _instance, bool _is_available) {
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::on_availability_cbk_forSwipe(): entered: service: "),
            DLT_INT(_service),
            DLT_CSTRING(", instance: "), DLT_INT(_instance),
            DLT_CSTRING(", is available:"), DLT_INT((int)_is_available));
    if (
            (CLUSTER_ANIMATED_SWIPE_SERVICE_ID == _service)
            && (CLUSTER_ANIMATED_SWIPE_INSTANCE_ID == _instance)
            && (true == _is_available)
            )
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::on_availability_cbk_forSwipe(): animated swipe is available "));
        ServiceAvailable_ClusterCombinationAvailable = true;
    }
    else
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::on_availability_cbk_forSwipe(): not interested in this service"));
        ServiceAvailable_ClusterCombinationAvailable = false;
    }
    _pCombinationServiceAvailabilityHandlerReference->NewDataReceivedFromClient(ServiceAvailable_ClusterCombinationAvailable);
    condition_.notify_one(); // this (might) be required if application is been stopped. But not sure at the moment if it's really required. The idea is to keep the application running since all services are disconnected.
                             // actually notify_one is also done by the caller function. No idea what is really required.
                             // ToDo: stop application scenario must be tested
}


void SomeIPGateway::on_state_cbk(vsomeip::state_type_e _state) {
    std::cout << "****************** SomeIPGateway::on_state_cbk(): enter"
            << "Application " << app_->get_name() << " is "
            << (_state == vsomeip::state_type_e::ST_REGISTERED ? "registered." : "deregistered.")
            << std::endl;

    DLT_LOG(SOMEIP, DLT_LOG_INFO,
            DLT_CSTRING("SomeIPGateway::state_cbk(): entered: Application "),
//            DLT_CSTRING(app_->get_name()),
            DLT_CSTRING("is registered:"),
            DLT_INT((int)_state));

    if (_state == vsomeip::state_type_e::ST_REGISTERED) {
        offerService();
        offerEvents();
        SomeIPApplicationIsRegistered = true;
    }
    else
    {
        std::cout << "SomeIPGateway::on_state_cbk(): not registered" << std::endl;
        DLT_LOG(SOMEIP, DLT_LOG_INFO,
                DLT_CSTRING("SomeIPGateway::state_cbk(): not registered"));

        SomeIPApplicationIsRegistered = false;

    }
    std::cout << "****************** SomeIPGateway::on_state_cbk(): exit" << std::endl;
}

vsomeip::byte_t SomeIPGateway::getPayloadFromCombinationResponseMessage(std::shared_ptr<vsomeip::message> pResponse)
{
    SOMEIP_NULL_POINTER_CHECK(pResponse);
    std::shared_ptr < vsomeip::payload > pPayload = pResponse->get_payload();
    SOMEIP_NULL_POINTER_CHECK(pPayload);
    vsomeip::byte_t* pPayloadDataVector = pPayload->get_data();
    SOMEIP_NULL_POINTER_CHECK(pPayloadDataVector);
    vsomeip::byte_t ClusterSwipeStatusValue = pPayloadDataVector[0];
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::getPayloadFromCombinationResponseMessage(): received status:"),
            DLT_INT(ClusterSwipeStatusValue),
            DLT_CSTRING(" from client:"), DLT_INT(pResponse->get_client()),
            DLT_CSTRING(" session:"), DLT_INT(pResponse->get_session())
            );
    return (ClusterSwipeStatusValue);
}

// This is the reponse message we receive, any time after cluster combination request was sent to Cluster.
// ToDo: refactoring required,  once we know whether whatever we sent to cluster works: Target is to get rid of everything what's message related
void SomeIPGateway::onClusterCombinationResponseMessage(const std::shared_ptr<vsomeip::message> &_response)
{
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::onClusterCombinationResponseMessage(): entered"));
    SOMEIP_NULL_POINTER_CHECK_NO_RET_VALUE(_response);
    DLT_LOG(SOMEIP,DLT_LOG_INFO,
            DLT_CSTRING("SomeIPGateway::onClusterCombinationResponseMessage(): SERVICE: Received message with Client/Session ["),
            DLT_INT(_response->get_client()), DLT_CSTRING("/"),
            DLT_INT(_response->get_session()), DLT_CSTRING("]"));

    vsomeip::byte_t ClusterSwipeStatusValue = getPayloadFromCombinationResponseMessage(_response);
    _pCombinationResponseHandlerReference->NewDataReceivedFromClient(ClusterSwipeStatusValue);
}

// ToDo: refactoring required,  once we know whether whatever we sent to cluster works: Target is to get rid of everything what's message related
void SomeIPGateway::onLayoutRequestNoReturnMessage(const std::shared_ptr<vsomeip::message> &_request) {
    std::cout << "********************** SomeIPGateway::onLayoutRequestNoReturnMessage(): entered" << std::endl;

    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::onLayoutRequestNoReturnMessage(): entered"));
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::onLayoutRequestNoReturnMessage(): Received message from cluster: Client(dec)/Session(dec) ["),
            DLT_INT(_request->get_client()), DLT_CSTRING("/"), DLT_INT(_request->get_session()), DLT_CSTRING("] instance: (dec) "),
            DLT_INT(_request->get_instance()), DLT_CSTRING(" protocol version: (dec) "), DLT_INT(_request->get_protocol_version()),
            DLT_CSTRING(" interface version: (dec) "), DLT_INT(_request->get_interface_version()), DLT_CSTRING(" request: (dec) "),
            DLT_INT(_request->get_request()), DLT_CSTRING(" method: (dec) "), DLT_INT(_request->get_method()));
    std::shared_ptr < vsomeip::payload > pPayload = _request->get_payload();
    vsomeip::byte_t* pLayoutDataVector = pPayload->get_data();
    vsomeip::byte_t RequestedLayout = pLayoutDataVector[0];
    if (nullptr != _pLayoutRequestHandlerReference) {
        _pLayoutRequestHandlerReference->NewDataReceivedFromClient((short) RequestedLayout);
    } else {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("NULL pointer exception: Layout Request Handler is not defined !!!"));
        std::cout << "NULL pointer exception: Layout Request Handler is not defined !!!" << std::endl;
    }
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::onLayoutRequestNoReturnMessage(): exit"));
    std::cout << "****************** SomeIPGateway::onLayoutRequestNoReturnMessage(): exit" << std::endl;

}
bool SomeIPGateway::checkWhetherStreamStatusShallBeSent(short StreamStatus)
{
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherStreamStatusShallBeSent(): entered"));
    bool SendStreamStatus = false;

    if (false == oSomeIPMessageSender.oEventQueue.isTableEmpty())
    {
        // do a defensive programming, we send anything we got
        SendStreamStatus = true;
//        if (StreamStatus!=oSomeIPMessageSender.oEventQueue.readElement(oSomeIPMessageSender.oEventQueue.size()))
//        {
//            SendStreamStatus = true;
//        }
//        else
//        {
//            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendVideoStreamStatusEvent: incoming stream status request already waiting to be sent"));
//            std::cout << "SomeIPGateway::sendVideoStreamStatusEvent: incoming stream status request already waiting to be sent" << std::endl;
//        }
    }
    else
    {
        SendStreamStatus = true;
    }
return (SendStreamStatus);
}

/**
 * Figure out whether a client has been registered for the event. If none is registered, event will not been sent.
 * We need this for debugging at the moment, to figure out, why an event is sometimes not sent to client.
 * @param ServiceID
 * @return
 */
bool SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(vsomeip::service_t ServiceID)
{
	DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(): entered  for service ID"),
			DLT_INT(ServiceID)
	);

    bool AtLeastOneClientIsRegistered = false;
    switch (ServiceID)
    {
    case CLUSTER_VIDEO_SERVICE_ID:
    {
        if (true == EventSubscription_MapInCLusterClientIsRegisteredForEvent)
        {
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(): a client is registered for map event"));
            AtLeastOneClientIsRegistered = true;
        }
        else
        {
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(): no client is registered for map event"));
        }
        break;
    }
    case CLUSTER_PICTURE_SERVICE_ID:
    {
        if (true == EventSubscription_ImageInCLusterClientIsRegisteredForEvent)
        {
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(): a client is registered for image event"));
            AtLeastOneClientIsRegistered = true;
        }
        else
        {
            DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(): no client is registered for image event"));
        }
        break;
    }
    default:
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(): client is unknown"));
        break;
    }
    }
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(): exit  for service ID"),
                    DLT_INT(ServiceID),
                    DLT_CSTRING("is a client registered:"), DLT_BOOL(AtLeastOneClientIsRegistered)
                    );

    return (AtLeastOneClientIsRegistered);
}

int SomeIPGateway::sendEvent(vsomeip::service_t serviceID, vsomeip::instance_t instanceID, vsomeip::event_t eventID, std::vector<vsomeip::byte_t> data) {
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendEvent(): entered  for service ID"),
                    DLT_INT(serviceID),
                    DLT_CSTRING("instanceID"), DLT_INT(instanceID),
                    DLT_CSTRING("eventID"), DLT_INT(eventID),
                    DLT_CSTRING("is application registered:"),DLT_BOOL(SomeIPApplicationIsRegistered)
                    );
    printf("SomeIPGateway::sendEvent entered: event for service ID: %d, instanceID: %d, eventID: %d, is application registered: %d\n",serviceID, instanceID, eventID, SomeIPApplicationIsRegistered);

    if ( true == ServiceOffer_MapInClusterIsOffered)
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendEvent(): service is offered, send event"));
        printf("SomeIPGateway::sendEvent(): service is offered, send event\n");

        std::shared_ptr < vsomeip::payload > payload = rtm_->create_payload();
        payload->set_data(data);

        checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(serviceID); // we use this for tracing only. Client should not request layout, if it's not been registered for event
//        sleep(5);
        SleepTimer Sleeper;
//        Sleeper.usleep(100);

        app_->notify(serviceID, instanceID, eventID, payload, true, true);// we want to send and event immediately and even value is not changed.
        bEventNotSent = FALSE;
    }
    else
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendEvent(): cant't send, service not offered"));
        _ServiceID = serviceID;
        _ServiceInstanceID = instanceID;
        _EventID = eventID;
        _oData = data;
        bEventNotSent = TRUE;
        printf("SomeIPGateway::sendEvent(): cant't send, service not offered and resend event once service available\n");

    }
    return 0;
}
int SomeIPGateway::offerEvent(vsomeip::service_t service_id, vsomeip::instance_t instanceID, int eventID, std::set<vsomeip::eventgroup_t> Groups) {
    DLT_LOG(SOMEIP, DLT_LOG_INFO,
                    DLT_CSTRING("SomeIPGateway::offerEvent(): entered: for offer service: service ID"),DLT_INT(service_id),
                    DLT_CSTRING("instanceID"), DLT_INT(instanceID),
                    DLT_CSTRING("eventID"), DLT_INT(eventID)
//                    DLT_CSTRING("group"),DLT_INT(Groups)
                    );
    std::cout << "SomeIPGateway::offerEvent entered\n" << std::endl;
    app_->offer_event(service_id, instanceID, eventID, Groups, true);
    return 0;
}
void SomeIPGateway::sendEvent(vsomeip::service_t ServiceID, vsomeip::instance_t InstanceID, int EventID, std::shared_ptr<vsomeip::payload> Payload) {
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendEvent(): entered, call gateway for event sending"),
                    DLT_CSTRING(" service ID"), DLT_INT(ServiceID),
                                        DLT_CSTRING("instanceID"), DLT_INT(InstanceID),
                                        DLT_CSTRING("eventID"), DLT_INT(EventID));
    //try sending an event to client
    if (nullptr!=pSomeIPGateway)
    {
        checkWhetherAtLeastOneClientHasBeenRegisteredForEvent(ServiceID); // we use this for tracing only. Client should not request layout, if it's not been registered for event
        pSomeIPGateway->app_->notify(ServiceID, InstanceID, EventID, Payload, true, true);  // we want to send and event immediately and even value is not changed.
    }
}
void SomeIPGateway::sendEvent(vsomeip::service_t ServiceID, vsomeip::instance_t InstanceID, int EventID, short StreamStatus) {
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendEvent(): entered, stream status received: service ID"),
                    DLT_INT(ServiceID),
                    DLT_CSTRING("instanceID"), DLT_INT(InstanceID),
                    DLT_CSTRING("eventID"), DLT_INT(EventID),
                    DLT_CSTRING("stream status to be sent"),DLT_INT(StreamStatus));

    std::shared_ptr < vsomeip::payload > pVideoStreamStatusPayload = pSomeIPGateway->rtm_->create_payload();
    std::vector < vsomeip::byte_t > oData;
    oData.push_back(StreamStatus);

    //try sending an event to client
    if (nullptr!=pSomeIPGateway)
    {
        pSomeIPGateway->sendEvent(ServiceID, InstanceID, EventID, oData);
    }
}

/**
 * implementation of someip_MessageSenderItf function
 * This is been called, once video status is been received from navigation service. It let us know whether the stream is been started or stopped.
 * The payload will be stored in a queue and the consumer thread will be informed about new data ready to be send to cluster.
 *
 * @param StreamStatus
 */
void SomeIPGateway::sendVideoStreamStatusEvent(short StreamStatus) {
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendVideoStreamStatusEvent to client: enter"));
#if defined (CLUSTER_EVENT_HANDLING_USE_THREAD_FOR_SENDING_EVENTS)
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendVideoStreamStatusEvent: use event thread: status to be sent:"), DLT_INT(StreamStatus));
    bool AddStatus = checkWhetherStreamStatusShallBeSent(StreamStatus);
    if (true == AddStatus)
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendVideoStreamStatusEvent: add incoming event to sent queue: event:"), DLT_INT(StreamStatus));

        oSomeIPMessageSender.storeElement(StreamStatus);
        oObserverThread.newSentEventRequestIncoming();
    }
#else
    std::cout << "SomeIPGateway::sendVideoStreamStatusEvent: send event with payload: " << StreamStatus << " to Cluster" << std::endl;
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendVideoStreamStatusEvent: send event with payload: "), DLT_INT(StreamStatus), DLT_CSTRING(" to Cluster"));
     sendEvent(CLUSTER_VIDEO_SERVICE_ID, CLUSTER_VIDEO_INSTANCE_ID, CLUSTER_VIDEO_STREAM_STATUS_EVENT_ID, StreamStatus);
#endif
}

void SomeIPGateway::sendIVIActionCombination(short TypeOfContents)
{
    DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendIVIActionCombination: send request with payload: "), DLT_INT(TypeOfContents), DLT_CSTRING(" to Cluster"));
    if (true==ServiceAvailable_ClusterCombinationAvailable)
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendIVIActionCombination: server is ready sent it!"));
        std::shared_ptr<vsomeip::message> pMessage;
        pMessage = rtm_->create_request();
        pMessage->set_service(CLUSTER_ANIMATED_SWIPE_SERVICE_ID);
        pMessage->set_instance(CLUSTER_ANIMATED_SWIPE_INSTANCE_ID);
        pMessage->set_method(CLUSTER_ANIMATED_SWIPE_IVI_ACTION_COMBINATION_METHOD_ID);
        pMessage->set_interface_version(MAJOR_VERSION);
        std::shared_ptr < vsomeip::payload > pPayload = rtm_->create_payload();
        std::vector<vsomeip::byte_t> oPayload;
        oPayload.push_back((vsomeip::byte_t)TypeOfContents);
        pPayload->set_data(oPayload);
        pMessage->set_payload(pPayload);
        pSomeIPGateway->app_->send(pMessage, true);

    }
    else
    {
        DLT_LOG(SOMEIP, DLT_LOG_INFO, DLT_CSTRING("SomeIPGateway::sendIVIActionCombination: server didn't offer the method, so sending skipped"));
    }
}

void SomeIPGateway::storeMediaImageExecutorReference(PictureExecutorItf* MediaImageExecutorReference) {
    m_MediaImageExecutorReference = MediaImageExecutorReference;
}

